/*
  ******************************************************************************
  * @file    app.c
  * @version V1.0.0
  * @date    2021
  * @brief   SPI demo source code.
  ******************************************************************************
*/
#include "app.h"


#define BUFFER_LENGTH    (256)

uint8_t gu8_TxBuffer[BUFFER_LENGTH];
uint8_t gu8_RxBuffer[BUFFER_LENGTH];

#define USERKEY_PORT	GPIOB
#define USERKEY_PIN		GPIO_PIN_9
GPIO_InitTypeDef    KEY_Handle;


/*********************************************************************************
* Function    : UserKEY_Init
* Description : User key initialize
* Input       : 
* Output      : 
**********************************************************************************/
void UserKEY_Init(void)
{
    KEY_Handle.Pin       = USERKEY_PIN;
    KEY_Handle.Mode      = GPIO_MODE_INPUT;
    KEY_Handle.Pull      = GPIO_PULLUP;
    KEY_Handle.Alternate = GPIO_FUNCTION_0;

    GPIO_Init(USERKEY_PORT, &KEY_Handle);
}

/*********************************************************************************
* Function    : UserKEY_Get
* Description : detection button is pressed
* Input       : 
* Output      : 
**********************************************************************************/
bool UserKEY_Get(void)
{
    if (Bit_RESET == GPIO_ReadInputDataBit(USERKEY_PORT, USERKEY_PIN)) 
    {
        System_Delay_MS(20);
        
        if (Bit_RESET == GPIO_ReadInputDataBit(USERKEY_PORT, USERKEY_PIN)) 
        {
            return true;
        }
    }

    return false;
}

/************************************************************************
 * function   : SPI_Master_Comm_Test
 * Description: SPI Master Communicate Test. 
 ************************************************************************/ 
void SPI_Master_Comm_Test(void)
{
    uint32_t i;
    
    uint32_t lu32_COM_OK  = 0;
    uint32_t lu32_COM_Err = 0;
    uint32_t lu32_DataLength = 50;

    printfS("SPI Master Test is Ready!!! \r\n");

    Spi_Gpio_Init();
    Spi_Init();
    
    do
    {        
        for (i = 0; i < BUFFER_LENGTH; i++)
        {
            gu8_TxBuffer[i] = i;
        }

        SPI_Transmit(gu8_TxBuffer, lu32_DataLength, 0);        
        System_Delay(50000);
        SPI_Receive(gu8_RxBuffer, lu32_DataLength, 0);
        
        for (i = 0; i < lu32_DataLength; i++)
        {
            if (gu8_TxBuffer[i] != gu8_RxBuffer[i]) 
            {
                printfS("There is one mistake : gu8_TxBuffer[%d] != gu8_RxBuffer[%d] \r\n", i, i);
                
                lu32_COM_Err++;
            }
            else 
            {
                lu32_COM_OK++;
            }
        }
        
        printfS("SPI Master Test OK count %d times \r\n",  lu32_COM_OK);
        printfS("SPI Master Test Err count %d times \r\n", lu32_COM_Err);

        lu32_COM_OK = 0;
        lu32_COM_Err = 0;
        
        while(false == UserKEY_Get());
        System_Delay_MS(500);
    }while(1);
    
    printfS("SPI Master Test End!!! \r\n");
}

/************************************************************************
 * function   : SPI_Master_DMA_Test
 * Description: SPI Master DMA Test.
 ************************************************************************/ 
void SPI_Master_DMA_Test(void)
{
    uint32_t i;

    uint32_t lu32_COM_OK  = 0;
    uint32_t lu32_COM_Err = 0;
    uint32_t lu32_DataLength = 50;

    printfS("SPI Master DMA Test is Ready!!! \r\n");

    Spi_Gpio_Init();
    Spi_Init();
    //Enable DMA Clock
    System_Module_Enable(EN_DMA);
    
    do
    {        
        for (i = 0; i < BUFFER_LENGTH; i++)
        {
            gu8_TxBuffer[i] = i;
        }

        SPI_Transmit_DMA(gu8_TxBuffer, lu32_DataLength);
        while (SPI_GetTxState() == SPI_TX_STATE_SENDING);
        System_Delay(500000);

        SPI_Receive_DMA(gu8_RxBuffer, lu32_DataLength);
        while (SPI_GetRxState() == SPI_RX_STATE_RECEIVING);
        System_Delay(500000);
        
        for (i = 0; i < lu32_DataLength; i++)
        {
            if (gu8_TxBuffer[i] != gu8_RxBuffer[i]) 
            {
                printfS("There is one mistake : gu8_TxBuffer[%d] != gu8_RxBuffer[%d] \r\n", i, i);
                
                lu32_COM_Err++;
            }
            else 
            {
                lu32_COM_OK++;
            }
        }
        
        printfS("SPI Master DMA Test OK count %d times \r\n",  lu32_COM_OK);
        printfS("SPI Master DMA Test Err count %d times \r\n", lu32_COM_Err);
        
        lu32_COM_OK = 0;
        lu32_COM_Err = 0;

        while(false == UserKEY_Get());
        System_Delay_MS(500);
    }while(1);

    printfS("SPI Master DMA Test End!!! \r\n");
}

/************************************************************************
 * function   : SPI_Master_FullDuplex_Test
 * Description: SPI Master Full Duplex Test. 
 ************************************************************************/ 
void SPI_Master_FullDuplex_Test(void)
{
    uint32_t i;
    
    uint32_t lu32_COM_OK  = 0;
    uint32_t lu32_COM_Err = 0;
    uint32_t lu32_DataLength = 50;

    printfS("SPI Master Full Duplex Test is Ready!!! \r\n");

    Spi_Gpio_Init();
    Spi_Init();
        
    do
    {
        for (i = 0; i < BUFFER_LENGTH; i++)
        {
            gu8_TxBuffer[i] = i;
        }
        
        SPI_TransmitReceive(gu8_TxBuffer, gu8_RxBuffer, lu32_DataLength, 0);
        
        for (i = 0; i < lu32_DataLength; i++)
        {
            if (gu8_TxBuffer[i] != (gu8_RxBuffer[i]-1)) 
            {
                printfS("There is one mistake : gu8_TxBuffer[%d]: 0x%02x != gu8_RxBuffer[%d]: 0x%02x \r\n", i,gu8_TxBuffer[i], i, gu8_RxBuffer[i]);
                
                lu32_COM_Err++;
            }
            else 
            {
                lu32_COM_OK++;
            }
        }
        
        printfS("SPI Master Full Duplex Test OK count %d times \r\n",  lu32_COM_OK);
        printfS("SPI Master Full Duplex Test Err count %d times \r\n", lu32_COM_Err);
        
        lu32_COM_OK = 0;
        lu32_COM_Err = 0;

        while(false == UserKEY_Get());
        System_Delay_MS(500);
    }while(1);
    
    printfS("SPI Master Full Duplex Test End!!! \r\n");
}


/************************For SPI NorFlash Test***************************/ 
extern SPI_InitTypeDef    spi_init;

/************************************************************************
 * function   : SPI_Init_Nor
 * Description: SPI Initiation For SPI NorFlash Test. 
 ************************************************************************/ 
void SPI_Init_Nor(void)
{
    /* Enable Clock */
    System_Module_Enable(EN_SPI3);
    
    spi_init.SPI_Mode               = SPI_MODE_MASTER;
    spi_init.SPI_CPOL               = SPI_CPOL_LOW;
    spi_init.SPI_CPHA               = SPI_CPHA_1EDGE;
    spi_init.SPI_X_Mode             = SPI_4X_MODE;
    spi_init.SPI_FirstBit           = SPI_FIRSTBIT_MSB;
    spi_init.SPI_BaudRatePrescaler  = SPI_BAUDRATE_PRESCALER_32;

    SPI_Init(SPI3, &spi_init);
}

/************************************************************************
 * function   : SPI_Nor_Flash_Test
 * Description: SPI_Nor_Flash_Test.
 ************************************************************************/ 
void SPI_Nor_Flash_Test(void)
{
    uint32_t i;
    
    uint16_t lu16_ID;
    uint32_t lu32_ErrCount = 0;
    uint32_t lu32_TestMode = 0;
    
    GPIO_InitTypeDef GPIO_Handle; 
    
    #define TEST_1X_MODE    (0)
    #define TEST_2X_MODE    (2)
    #define TEST_4X_MODE    (3)

    Spi_Gpio_Init();
    SPI_Init_Nor();
    
    /* CS USE Software Control */
    /*   SPI CS PortA Pin15    */
    GPIO_Handle.Pin            = GPIO_PIN_15;
    GPIO_Handle.Mode           = GPIO_MODE_OUTPUT_PP;
    GPIO_Handle.Pull           = GPIO_PULLUP;
    GPIO_Handle.Alternate      = GPIO_FUNCTION_0;
    GPIO_Init(GPIOA, &GPIO_Handle);

    __SPI_SET_1X_MODE();
    
    /* First Set CS HIGH */
    __SPI_CS_Release(); 
    
    System_Delay(50000);
    
    /* Please Select Test Mode */
    lu32_TestMode = TEST_1X_MODE;

    /*************************************** Test Prepare ***************************************/
    if (lu32_TestMode == TEST_1X_MODE) 
        printfS("SPI TEST_1X_MODE is Start!!! \r\n");
    else if (lu32_TestMode == TEST_2X_MODE) 
        printfS("SPI TEST_2X_MODE is Start!!! \r\n");
    else if (lu32_TestMode == TEST_4X_MODE) 
        printfS("SPI TEST_4X_MODE is Start!!! \r\n");

    /* Read Read Manufacture ID and Device ID */
    lu16_ID = IC_W25Qxx_Read_ID();

    printfS("Get Manufacture ID and Device ID : 0x%04X \r\n", lu16_ID);

    /* Erase Chip */
    IC_W25Qxx_EraseChip();
    
    IC_W25Qxx_Read_Data(gu8_RxBuffer, 0, BUFFER_LENGTH);

    for (i = 0; i < BUFFER_LENGTH; i++)
    {
        if (gu8_RxBuffer[i] != 0xFF) 
        {
            lu32_ErrCount++;
        } 
    }

    if (lu32_ErrCount) 
        printfS("Erase Chip Fail!!! \r\n");
    else 
        printfS("Erase Chip Success!!! \r\n");

    /* Clear Error Count */
    lu32_ErrCount = 0;
    
    for (i = 0; i < BUFFER_LENGTH; i++)
    {
        gu8_TxBuffer[i] = i;
    }
    
    /************************************* Test Prepare End **************************************/
    
    switch (lu32_TestMode)
    {
        case TEST_1X_MODE: 
        {
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 0,   256);
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 256, 256);
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 512, 256);
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 768, 256);
            
            IC_W25Qxx_Read_Data(gu8_RxBuffer, 0, BUFFER_LENGTH);
            
            for (i = 0; i < BUFFER_LENGTH; i++)
            {
                if (gu8_TxBuffer[i] != gu8_RxBuffer[i]) 
                {
                    lu32_ErrCount++;
                } 
            }
            
            if (lu32_ErrCount) 
                printfS("SPI TEST_1X_MODE is Fail!!! \r\n");
            else 
                printfS("SPI TEST_1X_MODE is Success!!! \r\n");
        }break;

        case TEST_2X_MODE: 
        {
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 0,   256);
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 256, 256);
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 512, 256);
            IC_W25Qxx_PageProgram(gu8_TxBuffer, 768, 256);
            
            IC_W25Qxx_Read_Dual_Output(gu8_RxBuffer, 0, BUFFER_LENGTH);
            
            for (i = 0; i < BUFFER_LENGTH; i++)
            {
                if (gu8_TxBuffer[i] != gu8_RxBuffer[i]) 
                {
                    lu32_ErrCount++;
                } 
            }

            if (lu32_ErrCount) 
                printfS("SPI TEST_2X_MODE is Fail!!! \r\n");
            else 
                printfS("SPI TEST_2X_MODE is Success!!! \r\n");
        }break;

        case TEST_4X_MODE: 
        {
            IC_W25Qxx_QuadConfig(true);
            
            IC_W25Qxx_PageProgram_Quad(gu8_TxBuffer, 0,   256);
            IC_W25Qxx_PageProgram_Quad(gu8_TxBuffer, 256, 256);
            IC_W25Qxx_PageProgram_Quad(gu8_TxBuffer, 512, 256);
            IC_W25Qxx_PageProgram_Quad(gu8_TxBuffer, 768, 256);
            
            IC_W25Qxx_Read_Quad_Output(gu8_RxBuffer, 0, BUFFER_LENGTH);
            
            for (i = 0; i < BUFFER_LENGTH; i++)
            {
                if (gu8_TxBuffer[i] != gu8_RxBuffer[i]) 
                {
                    lu32_ErrCount++;
                } 
            }

            if (lu32_ErrCount) 
                printfS("SPI TEST_4X_MODE is Fail!!! \r\n");
            else 
                printfS("SPI TEST_4X_MODE is Success!!! \r\n");

            IC_W25Qxx_QuadConfig(false);
        }break;

        default: break; 
    }
}

/************************************************************************
 * function   : SPI_Master_Test
 * Description: SPI Master Test. 
 ************************************************************************/ 
void SPI_Master_Test(enum_TEST_MODE_t fe_Mode)
{
    printfS("---------- SPI Master Test ----------\r\n");
    
    UserKEY_Init();
  
    printfS("---------- Please press the USR_PB button ----------\r\n");
    
    while(false == UserKEY_Get());
    System_Delay_MS(500);

    switch (fe_Mode)
    {
        case TEST_MASTER_COMM: 
            SPI_Master_Comm_Test();
            break; 
        
        case TEST_MASTER_DMA: 
            SPI_Master_DMA_Test();
            break;
        
        case TEST_MASTER_FULL_DUPLEX: 
            SPI_Master_FullDuplex_Test();
            break;
        
        case TEST_MASTER_NOR_FLASH: 
            SPI_Nor_Flash_Test();
            break;
        
        default: break;     
    }
}
