
/******************************************************************************
*@file  : spl_opa.c
*@brief : OPA SPL module driver.

*@ver   : 1.0.0
*@date  : 2022.10.20
******************************************************************************/

#include "fxx_std.h"

/******************************************************************************
*@brief : Initialize the OPA. 
*@note  : If the selected OPA is locked, de-initialization can't be performed.
*         To unlock the configuration, perform a module reset.
*@param : OPAMP_Selection: OPA selection,can be one of the following values:
*             @arg	OPAMP_Selection_OPAMP1
*             @arg	OPAMP_Selection_OPAMP2
*             @arg	OPAMP_Selection_OPAMP3
* @param  OPAMP_InitStruct: pointer to a OPAMP_InitTypeDef structure that contains
*         the configuration information for the specified OPAMP peripheral.
*@return: None
******************************************************************************/
void OPAMP_Init(uint32_t OPAMP_Selection, OPAMP_InitTypeDef* OPAMP_InitStruct)
{ 
    __IO uint32_t * tmpreg = 0;

  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));

    assert_param(IS_OPA_VINP(OPAMP_InitStruct->OPAMP_NonInvertingInput)); 
    assert_param(IS_OPA_VINN(OPAMP_InitStruct->OPAMP_InvertingInput)); 
    assert_param(IS_OPA_OUT(OPAMP_InitStruct->OutSel)); 
    assert_param(IS_OPA_HSM(OPAMP_InitStruct->Hsm)); 
    /*!< Get the OPAMPx_CSR register value */

    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    } 

    *tmpreg = ((OPAMP_InitStruct->Hsm & OPA_CSR_HSM_Msk) | \
                (OPAMP_InitStruct->OPAMP_InvertingInput & OPA_CSR_VINM0EN_Msk) | \
                (OPAMP_InitStruct->OPAMP_InvertingInput & OPA_CSR_MODESEL_Msk) | \
                (OPAMP_InitStruct->OPAMP_InvertingInput & OPA_CSR_POLSEL_Msk) | \
                (OPAMP_InitStruct->OPAMP_NonInvertingInput & OPA_CSR_VINPSEL_Msk) | \
                (OPAMP_InitStruct->OutSel & OPA_CSR_OUTSEL_Msk));

    return ;
}

/******************************************************************************
*@brief : de-Initialize the OPA. 
*@note  : If the selected OPA is locked, de-initialization can't be performed.
*         To unlock the configuration, perform a module reset.
*@param : OPAMP_Selection: OPA selection,can be one of the following values:
*             @arg	OPAMP_Selection_OPAMP1
*             @arg	OPAMP_Selection_OPAMP2
*             @arg	OPAMP_Selection_OPAMP3
*@return: None
******************************************************************************/
void OPAMP_DeInit(uint32_t OPAMP_Selection)
{ 
    __IO uint32_t * tmpreg;
  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));
    /*!< Get the OPAMPx_CSR register value */
    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    } 
    //disable opa
    CLEAR_BIT(*tmpreg, OPA_CSR_EN);
   
}

/******************************************************************************
*@brief : disable or enable the selected OPA.
*@param : OPAMP_Selection: OPA selection,can be one of the following values:
*             @arg	OPAMP_Selection_OPAMP1
*             @arg	OPAMP_Selection_OPAMP2
*             @arg	OPAMP_Selection_OPAMP3
* @param  NewState: new state of the OPAMP peripheral.
*          This parameter can be: ENABLE or DISABLE.
*@return: None
******************************************************************************/
void OPAMP_Cmd(uint32_t OPAMP_Selection, FunctionalState NewState)
{
    __IO uint32_t * tmpreg;
  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));
    /*!< Get the OPAMPx_CSR register value */
    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    } 
    
    if (NewState != DISABLE)
    {
        // Enable the opa
        SET_BIT(*tmpreg, OPA_CSR_EN); 
    }
    else
    {
        // Disable the opa
        CLEAR_BIT(*tmpreg, OPA_CSR_EN); 
    }

}

/******************************************************************************
*@brief : lock OPA.
*@param : OPAMP_Selection: OPA selection,can be one of the following values:
*             @arg	OPAMP_Selection_OPAMP1
*             @arg	OPAMP_Selection_OPAMP2
*             @arg	OPAMP_Selection_OPAMP3
*@return: None
******************************************************************************/
void OPAMP_LockConfig(uint32_t OPAMP_Selection)
{        
    __IO uint32_t * tmpreg;
  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));
    /*!< Get the OPAMPx_CSR register value */
    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    } 

    // lock the opa
    *tmpreg |= (uint32_t) OPA_CSR_LOCK; 
    
    return ;
}
/******************************************************************************
*@brief : Configure the trimming value of the OPAMP by factory(nvr value).
*@param : OPAMP_Selection: OPA selection,can be one of the following values:
*             @arg	OPAMP_Selection_OPAMP1
*             @arg	OPAMP_Selection_OPAMP2
*             @arg	OPAMP_Selection_OPAMP3
*@return: None
******************************************************************************/
void OPAMP_TrimFactory(uint32_t OPAMP_Selection)
{
    uint32_t nvrTrimValue;    
    __IO uint32_t * tmpreg;
  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));
    /*!< Get the OPAMPx_CSR register value */
    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    }
    //get the CSR register of OPAx
    if(OPAMP_Selection == OPAMP_Selection_OPAMP1)
    {
        //nvrTrimValue = HAL_EFLASH_READ_WORD(EFLASH_NVR2_OPA1_CAL_OSPN_ADDR);
    }
    else if(OPAMP_Selection == OPAMP_Selection_OPAMP2)
    {

        //nvrTrimValue = HAL_EFLASH_READ_WORD(EFLASH_NVR2_OPA2_CAL_OSPN_ADDR);        
    }    
    else if(OPAMP_Selection == OPAMP_Selection_OPAMP3)
    {
        //nvrTrimValue = HAL_EFLASH_READ_WORD(EFLASH_NVR2_OPA3_CAL_OSPN_ADDR);//SPLδʵ          
    } 
    else
    {
        return ;
    }
    /* Use the nvr Trim value */    
    if(((~(nvrTrimValue>>16))&0xFFFF) != (nvrTrimValue&0xFFFF)) return ;
    
    nvrTrimValue = nvrTrimValue & 0xFFFF;
    
    /* Enable the opa */
    *tmpreg |=  (uint32_t) OPA_CSR_EN;
    
    /* Trim the OPA_CSR_CAL_H N channel */ 
    *tmpreg |=  (uint32_t) OPA_CSR_CALNEN;
    *tmpreg &=  ~(uint32_t) OPA_CSR_CALPEN;

    MODIFY_REG(*tmpreg,OPA_CSR_TRIMOSN_Msk,((nvrTrimValue&0x1F)<<OPA_CSR_TRIMOSN_Pos));

    
    /* Trim the OPA_CSR_CAL_L P channel */    
    *tmpreg &=  ~(uint32_t) OPA_CSR_CALNEN;
    *tmpreg |=  (uint32_t) OPA_CSR_CALPEN;

    MODIFY_REG(*tmpreg,OPA_CSR_TRIMOSP_Msk,(((nvrTrimValue>>5)&0x1F)<<OPA_CSR_TRIMOSP_Pos));  

    
    //exit trim
    *tmpreg |=  (uint32_t) OPA_CSR_CALNEN;
    *tmpreg |=  (uint32_t) OPA_CSR_CALPEN;
    
}
/******************************************************************************
*@brief : Configure the trimming value of the OPAMP by software.
*@param : OPAMP_Selection: OPA selection,can be one of the following values:
*             @arg	OPAMP_Selection_OPAMP1
*             @arg	OPAMP_Selection_OPAMP2
*             @arg	OPAMP_Selection_OPAMP3
*@return: None
******************************************************************************/
void OPAMP_TrimSoftware(uint32_t OPAMP_Selection)
{
    uint32_t trimValue;    
    __IO uint32_t * tmpreg;
  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));
    /*!< Get the OPAMPx_CSR register value */
    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    }

    /* Enable the opa */
    *tmpreg |=  (uint32_t) OPA_CSR_EN;
    
    /* Trim the OPA_CSR_CAL_H N channel */ 
    *tmpreg |=  (uint32_t) OPA_CSR_CALNEN;
    *tmpreg &=  ~(uint32_t) OPA_CSR_CALPEN;

    for(trimValue=0;trimValue<31;trimValue++)
    {
        MODIFY_REG(*tmpreg,OPA_CSR_TRIMOSN_Msk,(trimValue)<<OPA_CSR_TRIMOSN_Pos);
        System_Delay(1000);
        if(READ_BIT( (*tmpreg),OPA_CSR_CALOUT)) break;
    }
    
    /* Trim the OPA_CSR_CAL_L P channel */    
    *tmpreg &=  ~(uint32_t) OPA_CSR_CALNEN;
    *tmpreg |=  (uint32_t) OPA_CSR_CALPEN;

    for(trimValue=0;trimValue<31;trimValue++)
    {
        MODIFY_REG(*tmpreg,OPA_CSR_TRIMOSP_Msk,(trimValue)<<OPA_CSR_TRIMOSP_Pos);
        System_Delay(1000);
        if(READ_BIT( (*tmpreg),OPA_CSR_CALOUT)) break;
    }
    //exit trim
    *tmpreg |=  (uint32_t) OPA_CSR_CALNEN;
    *tmpreg |=  (uint32_t) OPA_CSR_CALPEN;
}
void OPAMP_PGAConfig(uint32_t OPAMP_Selection, uint32_t OPAMP_PGAGain)
{
    __IO uint32_t * tmpreg;
  
    assert_param(IS_OPAMP_ALL_PERIPH(OPAMP_Selection));
    assert_param(IS_OPA_GAIN(OPAMP_PGAGain));
    
    /*!< Get the OPAMPx_CSR register value */
    tmpreg = (__IO uint32_t *) (OPA_BASE + OPAMP_Selection);
    //check lock status
    if(*tmpreg & OPA_CSR_LOCK)
    {
        return ;
    }  
    /* Reset the configuration bits */
    *tmpreg &= (uint32_t)(~OPA_CSR_GAINSEL_Msk);

    /* Set the new configuration */
    *tmpreg |= (uint32_t) (OPAMP_PGAGain);

}
void OPAMP_StructInit(OPAMP_InitTypeDef* OPAMP_InitStruct)
{
    OPAMP_InitStruct->OPAMP_NonInvertingInput = OPA_VINP_0;
    OPAMP_InitStruct->OPAMP_InvertingInput = OPA_VINN_SA_IO;
    OPAMP_InitStruct->Hsm = OPA_HSM_HIGH;
    OPAMP_InitStruct->OutSel = OPA_OUT_GPIO;
    
}
