/**
  *************** (C) COPYRIGHT 2025 aisinochip ************************
  * @file      startup_acm32f4.S
  * @brief     startup_acm32f4 Devices vector table for Atollic toolchain.
  *            This module performs:
  *                - Set the initial SP
  *                - Set the initial PC == Reset_Handler,
  *                - Set the vector table entries with the exceptions ISR address
  *                - Configure the clock system   
  *                - Branches to main in the C library (which eventually
  *                  calls main()).
  *            After Reset the Star-MC1 processor is in Thread mode,
  *            priority is Privileged, and the Stack is set to Main.
  ******************************************************************************
  */

  .syntax unified
  .cpu cortex-m33
  .fpu softvfp 
  .thumb

.global g_pfnVectors
.global Default_Handler

/* start address for the initialization values of the .data section.
defined in linker script */
.word _sidata
/* start address for the .data section. defined in linker script */
.word _sdata
/* end address for the .data section. defined in linker script */
.word _edata
/* start address for the .bss section. defined in linker script */
.word _sbss
/* end address for the .bss section. defined in linker script */
.word _ebss

/*.equ  BootRAM, 0xF108F85F */

/**
 * @brief  This is the code that gets called when the processor first
 *          starts execution following a reset event. Only the absolutely
 *          necessary set is performed, after which the application
 *          supplied main() routine is called.
 * @param  None
 * @retval : None
*/

  .section .text.Reset_Handler
  .weak Reset_Handler
  .type Reset_Handler, %function
Reset_Handler:

/* Call the clock system intitialization function.*/    
    bl  System_Core_Config
 
/* Copy the data segment initializers from flash to SRAM */
    ldr r0, =_sdata
    ldr r1, =_edata
    ldr r2, =_sidata
    movs r3, #0
    b LoopCopyDataInit

CopyDataInit:
    ldr r4, [r2, r3]
    str r4, [r0, r3]
    adds r3, r3, #4

LoopCopyDataInit:
    adds r4, r0, r3
    cmp r4, r1
    bcc CopyDataInit
  
/* Zero fill the bss segment. */
    ldr r2, =_sbss
    ldr r4, =_ebss
    movs r3, #0
    b LoopFillZerobss

FillZerobss:
    str  r3, [r2]
    adds r2, r2, #4

LoopFillZerobss:
    cmp r2, r4
    bcc FillZerobss

/* Call static constructors */
  /*  bl __libc_init_array */
  
/* Call the applications entry point.*/
    bl main
  
LoopForever:
    b LoopForever
    
.size Reset_Handler, .-Reset_Handler

/**
 * @brief  This is the code that gets called when the processor receives an
 *         unexpected interrupt.  This simply enters an infinite loop, preserving
 *         the system state for examination by a debugger.
 *
 * @param  None
 * @retval : None
*/
    .section .text.Default_Handler,"ax",%progbits
Default_Handler:
Infinite_Loop:
  b Infinite_Loop
  .size Default_Handler, .-Default_Handler
/******************************************************************************
*
* The minimal vector table for a Star-MC1.  Note that the proper constructs
* must be placed on this to ensure that it ends up at physical address
* 0x0000.0000.
*
******************************************************************************/
  .section .isr_vector,"a",%progbits
  .type g_pfnVectors, %object
  .size g_pfnVectors, .-g_pfnVectors


g_pfnVectors:
    .word _estack                           // top of stack 
    .word Reset_Handler                     // -14
    .word NMI_Handler                       // -13
    .word HardFault_Handler                 // -12
    .word MemManage_Handler                 // -12
    .word BusFault_Handler                  // -11
    .word UsageFault_Handler                // -10
    .word 0                                 // -9
    .word 0                                 // -8
    .word 0                                 // -7
    .word 0                                 // -6
    .word SVC_Handler                       // -5
    .word DebugMon_Handler                  // -4
    .word 0                                 // -3
    .word PendSV_Handler                    // -2
    .word SysTick_Handler                   // -0

/* External Interrupts */
    .word WDT_IRQHandler                    // 0
    .word RTC_IRQHandler                    // 1
    .word EFC_IRQHandler                    // 2
    .word GPIOAB_IRQHandler                 // 3
    .word GPIOCD_IRQHandler                 // 4
    .word EXIT_IRQHandler                   // 5
    .word SRAM_PAPITY_IRQHandler            // 6
    .word CLKRDY_IRQHandler                 // 7
    .word UART4_IRQHandler                  // 8
    .word DMA_IRQHandler                    // 9
    .word UART3_IRQHandler                  // 10
    .word RSV_IRQHandler                    // 11
    .word ADC_IRQHandler                    // 12
    .word TIM1_BRK_UP_TRG_COM_IRQHandler    // 13
    .word TIM1_CC_IRQHandler                // 14
    .word TIM2_IRQHandler                   // 15
    .word TIM3_IRQHandler                   // 16
    .word TIM6_IRQHandler                   // 17
    .word TIM7_IRQHandler                   // 18
    .word TIM14_IRQHandler                  // 19
    .word TIM15_IRQHandler                  // 20
    .word TIM16_IRQHandler                  // 21
    .word TIM17_IRQHandler                  // 22
    .word I2C1_IRQHandler                   // 23
    .word I2C2_IRQHandler                   // 24
    .word SPI1_IRQHandler                   // 25
    .word SPI2_IRQHandler                   // 26
    .word UART1_IRQHandler                  // 27
    .word UART2_IRQHandler                  // 28
    .word LPUART_IRQHandler                 // 29
    .word SPI3_IRQHandler                   // 30
    .word AES_IRQHandler                    // 31
    .word USB_IRQHandler                    // 32
    .word DAC_IRQHandler                    // 33
    .word I2S_IRQHandler                    // 34
    .word GPIOEF_IRQHandler                 // 35
    .word CAN1_IRQHandler                   // 36
    .word CAN2_IRQHandler                   // 37
    .word FPU_IRQHandler                    // 38
    .word TIM4_IRQHandler                   // 39
    .word SPI4_IRQHandler                   // 40

                          

/*******************************************************************************
*
* Provide weak aliases for each Exception handler to the Default_Handler.
* As they are weak aliases, any function with the same name will override
* this definition.
*
*******************************************************************************/

    .weak NMI_Handler
    .thumb_set NMI_Handler              ,Default_Handler    // -14
    
    .weak HardFault_Handler
    .thumb_set HardFault_Handler        ,Default_Handler    // -13
    
    .weak MemManage_Handler
    .thumb_set MemManage_Handler        ,Default_Handler    // -12
    
    .weak BusFault_Handler
    .thumb_set BusFault_Handler         ,Default_Handler    // -11
    
    .weak UsageFault_Handler
    .thumb_set UsageFault_Handler       ,Default_Handler    // -10
    
    .weak SVC_Handler
    .thumb_set SVC_Handler              ,Default_Handler    // -5
    
    .weak DebugMon_Handler
    .thumb_set DebugMon_Handler         ,Default_Handler    // -4
    
    .weak PendSV_Handler
    .thumb_set PendSV_Handler           ,Default_Handler    // -2
    
/*   weak defined in system_acm32f4.c 
    .weak SysTick_Handler
    .thumb_set SysTick_Handler         ,Default_Handler    // -1
*/    
    .weak WDT_IRQHandler
    .thumb_set WDT_IRQHandler           ,Default_Handler    // 0
    
    .weak RTC_IRQHandler
    .thumb_set RTC_IRQHandler           ,Default_Handler    // 1
    
    .weak EFC_IRQHandler
    .thumb_set EFC_IRQHandler           ,Default_Handler    // 2
    
    .weak GPIOAB_IRQHandler
    .thumb_set GPIOAB_IRQHandler        ,Default_Handler    // 3
    
    .weak GPIOCD_IRQHandler
    .thumb_set GPIOCD_IRQHandler        ,Default_Handler    // 4
    
    .weak EXIT_IRQHandler
    .thumb_set EXIT_IRQHandler          ,Default_Handler    // 5
    
    .weak SRAM_PAPITY_IRQHandler
    .thumb_set SRAM_PAPITY_IRQHandler   ,Default_Handler    // 6
    
    .weak CLKRDY_IRQHandler
    .thumb_set CLKRDY_IRQHandler        ,Default_Handler    // 7
    
    .weak UART4_IRQHandler
    .thumb_set UART4_IRQHandler         ,Default_Handler    // 8
    
    .weak DMA_IRQHandler
    .thumb_set DMA_IRQHandler           ,Default_Handler    // 9
    
    .weak UART3_IRQHandler
    .thumb_set UART3_IRQHandler         ,Default_Handler    // 10
    
    .weak RSV_IRQHandler
    .thumb_set RSV_IRQHandler           ,Default_Handler    // 11
    
    .weak ADC_IRQHandler
    .thumb_set ADC_IRQHandler           ,Default_Handler    // 12
    
    .weak TIM1_BRK_UP_TRG_COM_IRQHandler
    .thumb_set TIM1_BRK_UP_TRG_COM_IRQHandler,Default_Handler    // 13
    
    .weak TIM1_CC_IRQHandler
    .thumb_set TIM1_CC_IRQHandler       ,Default_Handler    // 14
    
    .weak TIM2_IRQHandler
    .thumb_set TIM2_IRQHandler          ,Default_Handler    // 15
    
    .weak TIM3_IRQHandler
    .thumb_set TIM3_IRQHandler          ,Default_Handler    // 16
    
    .weak TIM6_IRQHandler
    .thumb_set TIM6_IRQHandler          ,Default_Handler    // 17
    
    .weak TIM7_IRQHandler
    .thumb_set TIM7_IRQHandler          ,Default_Handler    // 18
    
    .weak TIM14_IRQHandler
    .thumb_set TIM14_IRQHandler         ,Default_Handler    // 19
    
    .weak TIM15_IRQHandler
    .thumb_set TIM15_IRQHandler         ,Default_Handler    // 20
    
    .weak TIM16_IRQHandler
    .thumb_set TIM16_IRQHandler         ,Default_Handler    // 21
    
    .weak TIM17_IRQHandler
    .thumb_set TIM17_IRQHandler         ,Default_Handler    // 22
    
    .weak I2C1_IRQHandler
    .thumb_set I2C1_IRQHandler          ,Default_Handler    // 23
    
    .weak I2C2_IRQHandler
    .thumb_set I2C2_IRQHandler          ,Default_Handler    // 24
    
    .weak SPI1_IRQHandler
    .thumb_set SPI1_IRQHandler          ,Default_Handler    // 25
    
    .weak SPI2_IRQHandler
    .thumb_set SPI2_IRQHandler          ,Default_Handler    // 26
    
    .weak UART1_IRQHandler
    .thumb_set UART1_IRQHandler         ,Default_Handler    // 27
    
    .weak UART2_IRQHandler
    .thumb_set UART2_IRQHandler         ,Default_Handler    // 28
    
    .weak LPUART_IRQHandler
    .thumb_set LPUART_IRQHandler        ,Default_Handler    // 29
    
    .weak SPI3_IRQHandler
    .thumb_set SPI3_IRQHandler          ,Default_Handler    // 30
    
    .weak AES_IRQHandler
    .thumb_set AES_IRQHandler           ,Default_Handler    // 31
    
    .weak USB_IRQHandler
    .thumb_set USB_IRQHandler           ,Default_Handler    // 32
    
    .weak DAC_IRQHandler
    .thumb_set DAC_IRQHandler           ,Default_Handler    // 33
    
    .weak I2S_IRQHandler
    .thumb_set I2S_IRQHandler           ,Default_Handler    // 34
    
    .weak GPIOEF_IRQHandler
    .thumb_set GPIOEF_IRQHandler        ,Default_Handler    // 35
    
    .weak CAN1_IRQHandler
    .thumb_set CAN1_IRQHandler          ,Default_Handler    // 36
    
    .weak CAN2_IRQHandler
    .thumb_set CAN2_IRQHandler          ,Default_Handler    // 37
    
    .weak FPU_IRQHandler
    .thumb_set FPU_IRQHandler           ,Default_Handler    // 38
    
    .weak TIM4_IRQHandler
    .thumb_set TIM4_IRQHandler          ,Default_Handler    // 39
    
    .weak SPI4_IRQHandler
    .thumb_set EXTI15_10_IRQHandler     ,Default_Handler    // 40


/*****************************END OF FILE****/

