
/******************************************************************************
*@file  : app.c
*@brief : application
******************************************************************************/

#include "app.h"


extern volatile uint32_t gu32_SystemCount;

volatile uint32_t g_IRQFlag = 0;

/******************************************************************************
*@brief : WDT initialization 
*@param : None
*@return: None
******************************************************************************/
void WDT_Init(uint32_t mode)
{
    uint32_t timer_clk_hz, wdt_time;
    if (WDT_MODE_RESET == mode) 
    {
        printfS("start to test WDT, reset mode: 5s period! \n");  
    }
    else
    {
        printfS("start to test WDT, interrupt mode: 5s period! \n");     
    }

    timer_clk_hz = System_Get_APBClock();
    wdt_time = timer_clk_hz / 128;    

    WDT_DeInit();
    /* Enable WDT PCLK */
    SCU->IPCKENR2 |= (1 << 2);

    WDT_SetPrescaler(WDT_CLOCK_PRESCALER_128);
    
    WDT_SetMode(mode);
    
    WDT_SetLoadValue(wdt_time);   // 1000ms
    WDT_SetITClrTime(5 * (wdt_time / 1000));    // 50ms
    
    WDT_Cmd(WDT_STATE_ENABLE);
    
     
}  

/******************************************************************************
*@brief : IWDT interrup test 
*@param : None
*@return: None
******************************************************************************/
void WDT_Interrupt_Test(void)
{
    WDT_Init(WDT_MODE_INTERRUPT);
    WDT_ITCmd(WDT_INTERRUPT_ENABLE);
    // Enable NVIC IRQ
    NVIC_ClearPendingIRQ(WDT_IRQn);
    NVIC_SetPriority(WDT_IRQn, 0x00);
    NVIC_EnableIRQ(WDT_IRQn);
    
    while(1)
    {
        if (g_IRQFlag)
        {
            printfS("WDT IRQ Occurs!\n"); 
            g_IRQFlag = 0;
        } 
    }
}

/******************************************************************************
*@brief : WDT reset test 
*@param : None
*@return: None
******************************************************************************/
void WDT_Reset_Test(void)
{
    uint8_t cnt;
    IWDT_ResetCmd(WDT_STATE_ENABLE);
    WDT_Init(WDT_MODE_RESET);
    if((SCU->RSR & BIT1) == BIT1)
    {
        SCU->RSR = BIT16;
        printfS("Last Reset Reason: WDT\r\n");         
    }
    cnt = 0;    
    while(1)
    {
        /* ڸλʱ䵽ǰ150msˢWDT */
        System_Delay_MS(850);
        if(cnt++ < 10)
        {
            WDT_FeedDog(); 
            printfS("[%d]feed watchdog time : %u\r\n", cnt, gu32_SystemCount);    
        }
        else
        {
            printfS("[%d]: %u\r\n", cnt, gu32_SystemCount);
        }            
    } 
}






/******************************************************************************
*@brief : WDT test
*@param : test_mode: Test mode selection
*@return: none
******************************************************************************/
void WDT_Test(Test_mode_t test_mode)
{  
    switch(test_mode)
    {
        case Test_WDT_Reset:
        {
            WDT_Reset_Test();
        }break;
        case Test_WDT_IRQ:
        {
            WDT_Interrupt_Test();
        }break;
        default:
        {
            printfS("This application does not support\r\n");
        }break;
    }
}

/******************************************************************************
* @brief : app test.
* @param : None
* @return: None
******************************************************************************/
void APP_Test(void)
{
    /*
    Test_WDT_Reset,
    Test_WDT_IRQ,
    */
    WDT_Test(Test_WDT_Reset);

}










