
/******************************************************************************
*@file  : fxx_lpuart.h
*@brief : Header file of LPUART SPL module.
******************************************************************************/

#ifndef __FXX_LPUART_H__
#define __FXX_LPUART_H__

#include "fxx_std.h"
/***************  Bits definition for LPUART_RXDR register  ******************/

#define LPUART_RXDR_RXDATA_Pos                  ( 0U )
#define LPUART_RXDR_RXDATA_Msk                  ( 0xFFUL << LPUART_RXDR_RXDATA_Pos )
#define LPUART_RXDR_RXDATA                      ( LPUART_RXDR_RXDATA_Msk )


/***************  Bits definition for LPUART_TXDR register  ******************/

#define LPUART_TXDR_TXDATA_Pos                  ( 0U )
#define LPUART_TXDR_TXDATA_Msk                  ( 0xFFUL << LPUART_TXDR_TXDATA_Pos )
#define LPUART_TXDR_TXDATA                      ( LPUART_TXDR_TXDATA_Msk )


/***************  Bits definition for LPUART_LCR register  *******************/

#define LPUART_LCR_TXPOL_Pos                    ( 9U )
#define LPUART_LCR_TXPOL_Msk                    ( 0x1UL << LPUART_LCR_TXPOL_Pos )
#define LPUART_LCR_TXPOL                        ( LPUART_LCR_TXPOL_Msk )

#define LPUART_LCR_RXPOL_Pos                    ( 8U )
#define LPUART_LCR_RXPOL_Msk                    ( 0x1UL << LPUART_LCR_RXPOL_Pos )
#define LPUART_LCR_RXPOL                        ( LPUART_LCR_RXPOL_Msk )

#define LPUART_LCR_WKCK_Pos                     ( 7U )
#define LPUART_LCR_WKCK_Msk                     ( 0x1UL << LPUART_LCR_WKCK_Pos )
#define LPUART_LCR_WKCK                         ( LPUART_LCR_WKCK_Msk )

#define LPUART_LCR_RXWKS_Pos                    ( 5U )
#define LPUART_LCR_RXWKS_Msk                    ( 0x3UL << LPUART_LCR_RXWKS_Pos )
#define LPUART_LCR_RXWKS                        ( LPUART_LCR_RXWKS_Msk )
#define LPUART_LCR_RXWKS_0                      ( 0x1UL << LPUART_LCR_RXWKS_Pos )
#define LPUART_LCR_RXWKS_1                      ( 0x2UL << LPUART_LCR_RXWKS_Pos )

#define LPUART_LCR_WLEN_Pos                     ( 4U )
#define LPUART_LCR_WLEN_Msk                     ( 0x1UL << LPUART_LCR_WLEN_Pos )
#define LPUART_LCR_WLEN                         ( LPUART_LCR_WLEN_Msk )

#define LPUART_LCR_STP2_Pos                     ( 3U )
#define LPUART_LCR_STP2_Msk                     ( 0x1UL << LPUART_LCR_STP2_Pos )
#define LPUART_LCR_STP2                         ( LPUART_LCR_STP2_Msk )

#define LPUART_LCR_EPS_Pos                      ( 2U )
#define LPUART_LCR_EPS_Msk                      ( 0x1UL << LPUART_LCR_EPS_Pos )
#define LPUART_LCR_EPS                          ( LPUART_LCR_EPS_Msk )

#define LPUART_LCR_SPS_Pos                      ( 1U )
#define LPUART_LCR_SPS_Msk                      ( 0x1UL << LPUART_LCR_SPS_Pos )
#define LPUART_LCR_SPS                          ( LPUART_LCR_SPS_Msk )

#define LPUART_LCR_PEN_Pos                      ( 0U )
#define LPUART_LCR_PEN_Msk                      ( 0x1UL << LPUART_LCR_PEN_Pos )
#define LPUART_LCR_PEN                          ( LPUART_LCR_PEN_Msk )


/***************  Bits definition for LPUART_CR register  ********************/

#define LPUART_CR_DMAEN_Pos                     ( 2U )
#define LPUART_CR_DMAEN_Msk                     ( 0x1UL << LPUART_CR_DMAEN_Pos )
#define LPUART_CR_DMAEN                         ( LPUART_CR_DMAEN_Msk )

#define LPUART_CR_TXE_Pos                       ( 1U )
#define LPUART_CR_TXE_Msk                       ( 0x1UL << LPUART_CR_TXE_Pos )
#define LPUART_CR_TXE                           ( LPUART_CR_TXE_Msk )

#define LPUART_CR_RXE_Pos                       ( 0U )
#define LPUART_CR_RXE_Msk                       ( 0x1UL << LPUART_CR_RXE_Pos )
#define LPUART_CR_RXE                           ( LPUART_CR_RXE_Msk )


/***************  Bits definition for LPUART_IBAUD register  *****************/

#define LPUART_IBAUD_RXSAM_Pos                  ( 8U )
#define LPUART_IBAUD_RXSAM_Msk                  ( 0xFFUL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM                      ( LPUART_IBAUD_RXSAM_Msk )
#define LPUART_IBAUD_RXSAM_0                    ( 0x1UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_1                    ( 0x2UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_2                    ( 0x4UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_3                    ( 0x8UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_4                    ( 0x10UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_5                    ( 0x20UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_6                    ( 0x40UL << LPUART_IBAUD_RXSAM_Pos )
#define LPUART_IBAUD_RXSAM_7                    ( 0x80UL << LPUART_IBAUD_RXSAM_Pos )

#define LPUART_IBAUD_IBAUD_Pos                  ( 0U )
#define LPUART_IBAUD_IBAUD_Msk                  ( 0xFFUL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD                      ( LPUART_IBAUD_IBAUD_Msk )
#define LPUART_IBAUD_IBAUD_0                    ( 0x1UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_1                    ( 0x2UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_2                    ( 0x4UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_3                    ( 0x8UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_4                    ( 0x10UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_5                    ( 0x20UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_6                    ( 0x40UL << LPUART_IBAUD_IBAUD_Pos )
#define LPUART_IBAUD_IBAUD_7                    ( 0x80UL << LPUART_IBAUD_IBAUD_Pos )


/***************  Bits definition for LPUART_FBAUD register  *****************/

#define LPUART_FBAUD_FBAUD_Pos                  ( 0U )
#define LPUART_FBAUD_FBAUD_Msk                  ( 0xFFFUL << LPUART_FBAUD_FBAUD_Pos )
#define LPUART_FBAUD_FBAUD                      ( LPUART_FBAUD_FBAUD_Msk )


/***************  Bits definition for LPUART_IE register  ********************/

#define LPUART_IE_STARTIE_Pos                   ( 9U )
#define LPUART_IE_STARTIE_Msk                   ( 0x1UL << LPUART_IE_STARTIE_Pos )
#define LPUART_IE_STARTIE                       ( LPUART_IE_STARTIE_Msk )

#define LPUART_IE_MATCHIE_Pos                   ( 8U )
#define LPUART_IE_MATCHIE_Msk                   ( 0x1UL << LPUART_IE_MATCHIE_Pos )
#define LPUART_IE_MATCHIE                       ( LPUART_IE_MATCHIE_Msk )

#define LPUART_IE_RXOVIE_Pos                    ( 5U )
#define LPUART_IE_RXOVIE_Msk                    ( 0x1UL << LPUART_IE_RXOVIE_Pos )
#define LPUART_IE_RXOVIE                        ( LPUART_IE_RXOVIE_Msk )

#define LPUART_IE_FEIE_Pos                      ( 4U )
#define LPUART_IE_FEIE_Msk                      ( 0x1UL << LPUART_IE_FEIE_Pos )
#define LPUART_IE_FEIE                          ( LPUART_IE_FEIE_Msk )

#define LPUART_IE_PEIE_Pos                      ( 3U )
#define LPUART_IE_PEIE_Msk                      ( 0x1UL << LPUART_IE_PEIE_Pos )
#define LPUART_IE_PEIE                          ( LPUART_IE_PEIE_Msk )

#define LPUART_IE_TXEIE_Pos                     ( 2U )
#define LPUART_IE_TXEIE_Msk                     ( 0x1UL << LPUART_IE_TXEIE_Pos )
#define LPUART_IE_TXEIE                         ( LPUART_IE_TXEIE_Msk )

#define LPUART_IE_TCIE_Pos                      ( 1U )
#define LPUART_IE_TCIE_Msk                      ( 0x1UL << LPUART_IE_TCIE_Pos )
#define LPUART_IE_TCIE                          ( LPUART_IE_TCIE_Msk )

#define LPUART_IE_RXIE_Pos                      ( 0U )
#define LPUART_IE_RXIE_Msk                      ( 0x1UL << LPUART_IE_RXIE_Pos )
#define LPUART_IE_RXIE                          ( LPUART_IE_RXIE_Msk )


/***************  Bits definition for LPUART_SR register  ********************/

#define LPUART_SR_STARTIF_Pos                   ( 9U )
#define LPUART_SR_STARTIF_Msk                   ( 0x1UL << LPUART_SR_STARTIF_Pos )
#define LPUART_SR_STARTIF                       ( LPUART_SR_STARTIF_Msk )

#define LPUART_SR_MATCHIF_Pos                   ( 8U )
#define LPUART_SR_MATCHIF_Msk                   ( 0x1UL << LPUART_SR_MATCHIF_Pos )
#define LPUART_SR_MATCHIF                       ( LPUART_SR_MATCHIF_Msk )

#define LPUART_SR_TXOVF_Pos                     ( 7U )
#define LPUART_SR_TXOVF_Msk                     ( 0x1UL << LPUART_SR_TXOVF_Pos )
#define LPUART_SR_TXOVF                         ( LPUART_SR_TXOVF_Msk )

#define LPUART_SR_RXF_Pos                       ( 6U )
#define LPUART_SR_RXF_Msk                       ( 0x1UL << LPUART_SR_RXF_Pos )
#define LPUART_SR_RXF                           ( LPUART_SR_RXF_Msk )

#define LPUART_SR_RXOVIF_Pos                    ( 5U )
#define LPUART_SR_RXOVIF_Msk                    ( 0x1UL << LPUART_SR_RXOVIF_Pos )
#define LPUART_SR_RXOVIF                        ( LPUART_SR_RXOVIF_Msk )

#define LPUART_SR_FEIF_Pos                      ( 4U )
#define LPUART_SR_FEIF_Msk                      ( 0x1UL << LPUART_SR_FEIF_Pos )
#define LPUART_SR_FEIF                          ( LPUART_SR_FEIF_Msk )

#define LPUART_SR_PEIF_Pos                      ( 3U )
#define LPUART_SR_PEIF_Msk                      ( 0x1UL << LPUART_SR_PEIF_Pos )
#define LPUART_SR_PEIF                          ( LPUART_SR_PEIF_Msk )

#define LPUART_SR_TXEIF_Pos                     ( 2U )
#define LPUART_SR_TXEIF_Msk                     ( 0x1UL << LPUART_SR_TXEIF_Pos )
#define LPUART_SR_TXEIF                         ( LPUART_SR_TXEIF_Msk )

#define LPUART_SR_TCIF_Pos                      ( 1U )
#define LPUART_SR_TCIF_Msk                      ( 0x1UL << LPUART_SR_TCIF_Pos )
#define LPUART_SR_TCIF                          ( LPUART_SR_TCIF_Msk )

#define LPUART_SR_RXIF_Pos                      ( 0U )
#define LPUART_SR_RXIF_Msk                      ( 0x1UL << LPUART_SR_RXIF_Pos )
#define LPUART_SR_RXIF                          ( LPUART_SR_RXIF_Msk )


/***************  Bits definition for LPUART_ADDR register  ******************/

#define LPUART_ADDR_ADDR_Pos                    ( 0U )
#define LPUART_ADDR_ADDR_Msk                    ( 0xFFUL << LPUART_ADDR_ADDR_Pos )
#define LPUART_ADDR_ADDR                        ( LPUART_ADDR_ADDR_Msk )


/** 
  * @brief  LPUART Init structure definition
  */
typedef struct
{
    uint32_t    BaudRate;           /*!< This member configures the UART communication baud rate.
                                         This parameter can be a value of @ref LPUART_BaudRate. */

    uint32_t    WordLength;         /*!< Specifies the number of data bits transmitted or received in a frame.
                                         This parameter can be a value of @ref LPUART_WordLength. */
          
    uint32_t    StopBits;           /*!< Specifies the number of stop bits transmitted.
                                         This parameter can be a value of @ref LPUART_StopBits. */

    uint32_t    Parity;             /*!< Specifies the parity mode.
                                         This parameter can be a value of @ref LPUART_Parity*/

    uint32_t    Mode;               /*!< Specifies whether the Receive or Transmit mode is enabled or disabled.
                                         This parameter can be a value of @ref LPUART_Mode. */

} LPUART_InitTypeDef;


  
/** @defgroup LPUART_Mode
  * @{
  */
  
#define LPUART_MODE_TX                      ( LPUART_CR_TXE )
#define LPUART_MODE_RX                      ( LPUART_CR_RXE )
#define LPUART_MODE_TXRX                    ( LPUART_CR_TXE | LPUART_CR_RXE )

/**
  * @}
  */
  
/** @defgroup LPUART_ClockSource
  * @{
  */
#define SCU_CCR2_LPUART_CLKSEL_RC32K        (0 << 13)
#define SCU_CCR2_LPUART_CLKSEL_XTL          (1 << 13)
#define SCU_CCR2_LPUART_CLKSEL_PLL          (2 << 13)

#define SCU_CCR2_LPUART_CLK_PLL_DIV4        (0 <<11)
#define SCU_CCR2_LPUART_CLK_PLL_DIV8        (1 <<11)
#define SCU_CCR2_LPUART_CLK_PLL_DIV16       (2 <<11)
#define SCU_CCR2_LPUART_CLK_PLL_DIV32       (3 <<11)


#define LPUART_CLOCKSOURCE_RC32K            (0 << 13)
#define LPUART_CLOCKSOURCE_XTL              (1 << 13)
#define LPUART_CLOCKSOURCE_PCLK_DIV4        ((2 << 13) | (0 <<11))
#define LPUART_CLOCKSOURCE_PCLK_DIV8        ((2 << 13) | (1 <<11))
#define LPUART_CLOCKSOURCE_PCLK_DIV16       ((2 << 13) | (2 <<11))
#define LPUART_CLOCKSOURCE_PCLK_DIV32       ((2 << 13) | (3 <<11))

/**
  * @}
  */

/** @defgroup LPUART_WordLength
  * @{
  */
  
#define LPUART_WORDLENGTH_7B                ( LPUART_LCR_WLEN )
#define LPUART_WORDLENGTH_8B                ( 0U )

/**
  * @}
  */

/** @defgroup LPUART_StopBits
  * @{
  */
  
#define LPUART_STOPBITS_1B                  ( 0U )
#define LPUART_STOPBITS_2B                  ( LPUART_LCR_STP2 )

/**
  * @}
  */
  
/** @defgroup LPUART_Parity
  * @{
  */
  
#define LPUART_PARITY_NONE                  ( 0U )
#define LPUART_PARITY_ODD                   ( LPUART_LCR_PEN )
#define LPUART_PARITY_EVEN                  ( LPUART_LCR_PEN | LPUART_LCR_EPS )
#define LPUART_PARITY_0                     ( LPUART_LCR_PEN | LPUART_LCR_SPS )
#define LPUART_PARITY_1                     ( LPUART_LCR_PEN | LPUART_LCR_SPS | LPUART_LCR_EPS )

/**
  * @}
  */

/** @defgroup LPUART_WakeupMode
  * @{
  */
  
#define LPUART_WAKEUPMODE_NONE              ( LPUART_LCR_RXWKS )
#define LPUART_WAKEUPMODE_STARTBIT          ( 0U )
#define LPUART_WAKEUPMODE_ONEBYTENOCHECK    ( LPUART_LCR_RXWKS_0 )
#define LPUART_WAKEUPMODE_ONEBYTECHECK      ( LPUART_LCR_RXWKS_0 | LPUART_LCR_WKCK )
#define LPUART_WAKEUPMODE_ADDRNOCHECK       ( LPUART_LCR_RXWKS_1 )
#define LPUART_WAKEUPMODE_ADDRCHECK         ( LPUART_LCR_RXWKS_1 | LPUART_LCR_WKCK)

/**
  * @}
  */

/** @defgroup LPUART_IT
  * @{
  */
  
#define LPUART_IT_START                     ( LPUART_IE_STARTIE )
#define LPUART_IT_MATCH                     ( LPUART_IE_MATCHIE )
#define LPUART_IT_RXOV                      ( LPUART_IE_RXOVIE )
#define LPUART_IT_FE                        ( LPUART_IE_FEIE )
#define LPUART_IT_PE                        ( LPUART_IE_PEIE )
#define LPUART_IT_TXE                       ( LPUART_IE_TXEIE )
#define LPUART_IT_TC                        ( LPUART_IE_TCIE )
#define LPUART_IT_RX                        ( LPUART_IE_RXIE )

/**
  * @}
  */

/** 
  * @brief  
  */

#define LPUART_IT_MASK                      ( LPUART_IT_START | LPUART_IT_MATCH | \
                                              LPUART_IT_RXOV | LPUART_IT_FE | \
                                              LPUART_IT_PE | LPUART_IT_TXE | \
                                              LPUART_IT_TC | LPUART_IT_RX)

/** @defgroup LPUART_Flags
  * @{
  */

#define LPUART_FLAG_STARTIF                 ( LPUART_SR_STARTIF )
#define LPUART_FLAG_MATCHIF                 ( LPUART_SR_MATCHIF )
#define LPUART_FLAG_TXOVF                   ( LPUART_SR_TXOVF )
#define LPUART_FLAG_RXF                     ( LPUART_SR_RXF )
#define LPUART_FLAGR_RXOVIF                 ( LPUART_SR_RXOVIF )
#define LPUART_FLAG_FEIF                    ( LPUART_SR_FEIF )
#define LPUART_FLAG_PEIF                    ( LPUART_SR_PEIF )
#define LPUART_FLAG_TXEIF                   ( LPUART_SR_TXEIF )
#define LPUART_FLAG_TCIF                    ( LPUART_SR_TCIF )
#define LPUART_FLAG_RXIF                    ( LPUART_SR_RXIF )

/**
  * @}
  */

/** 
  * @brief  
  */

#define LPUART_IT_FLAG_MASK                 ( LPUART_FLAG_STARTIF | LPUART_FLAG_MATCHIF | \
                                              LPUART_FLAGR_RXOVIF | LPUART_FLAG_FEIF | \
                                              LPUART_FLAG_PEIF | LPUART_FLAG_TXEIF | \
                                              LPUART_FLAG_TCIF | LPUART_FLAG_RXIF)

#define LPUART_FLAG_MASK                    ( LPUART_FLAG_TXOVF | \
                                              LPUART_FLAG_RXF | LPUART_IT_FLAG_MASK)

#define LPUART_CLEAR_FLAG_MASK              ( LPUART_FLAG_STARTIF | LPUART_FLAG_MATCHIF | \
                                              LPUART_FLAG_TXOVF | LPUART_FLAGR_RXOVIF | \
                                              LPUART_FLAG_FEIF | LPUART_FLAG_PEIF | \
                                              LPUART_FLAG_TCIF | LPUART_FLAG_RXIF)
    

/** @defgroup  LPUART Private Macros
  * @{
  */
  
#define __LPUART_WRITE_TXDATA(__LPUART__, __TXDATA__)       ((__LPUART__)->TXDR = __TXDATA__)

#define __LPUART_READ_RXDATA(__LPUART__, __RXDATA__)        ((__LPUART__)->RXDR)

#define __LPUART_ENABLE_DMA(__LPUART__)                     ((__LPUART__)->CR |= LPUART_CR_DMAEN)

#define __LPUART_DISABLE_DMA(__LPUART__)                    ((__LPUART__)->CR &= ~LPUART_CR_DMAEN)

#define __LPUART_ENABLE_TX(__LPUART__)                      ((__LPUART__)->CR |= LPUART_CR_TXE)

#define __LPUART_DISABLE_TX(__LPUART__)                     ((__LPUART__)->CR &= ~LPUART_CR_TXE)

#define __LPUART_ENABLE_RX(__LPUART__)                      ((__LPUART__)->CR |= LPUART_CR_RXE)

#define __LPUART_DISABLE_RX(__LPUART__)                     ((__LPUART__)->CR &= ~LPUART_CR_RXE)

#define __LPUART_ENABLE_IT(__LPUART__, __INT__)             ((__LPUART__)->IE |= (__INTERRUPT__) & LPUART_IT_MASK)

#define __LPUART_DISABLE_IT(__LPUART__, __INT__)            ((__LPUART__)->IE &= ~((__INTERRUPT__) & LPUART_IT_MASK))

#define __LPUART_GET_FLAG(__LPUART__, __FLAG__)             ((__LPUART__)->SR & ((__FLAG__) & LPUART_FLAG_MASK))

#define __LPUART_GET_IT_FLAG(__LPUART__, __FLAG__)          ((__LPUART__)->SR & ((__FLAG__) & LPUART_IT_FLAG_MASK))

#define __LPUART_GET_IT_SOURCE(__LPUART__, __INT__)         ((__LPUART__)->IE & ((__INTERRUPT__) & LPUART_IT_MASK))

#define __LPUART_CLEAR_FLAG(__LPUART__, __FLAG__)           ((__LPUART__)->SR = ((__FLAG__) & LPUART_CLEAR_FLAG_MASK))


/** @defgroup  LPUART Private Macros
  * @{
  */
  
#define IS_LPUART_ALL_PERIPH(__PERIPH__)                    ((__PERIPH__) == LPUART)

#define IS_LPUART_FUNCTIONAL_STATE(_STATE_)                 (((_STATE_) == ENABLE)      || \
                                                             ((_STATE_) == DISABLE))
                                                 

#define IS_LPUART_BAUDRATE(__BAUDRATE__)                    ((__BAUDRATE__) != 0)

#define IS_LPUART_IBAUD(__IBAUD__)                          (((__IBAUD__) >= 2U) && ((__IBAUD__) <= 254U))

#define IS_LPUART_CLOCKSOURCE(__SOURCE__)                   (((__SOURCE__) == LPUART_CLOCKSOURCE_RC32K)     || \
                                                             ((__SOURCE__) == LPUART_CLOCKSOURCE_XTL)       || \
                                                             ((__SOURCE__) == LPUART_CLOCKSOURCE_PCLK_DIV4) || \
                                                             ((__SOURCE__) == LPUART_CLOCKSOURCE_PCLK_DIV8) || \
                                                             ((__SOURCE__) == LPUART_CLOCKSOURCE_PCLK_DIV16)|| \
                                                             ((__SOURCE__) == LPUART_CLOCKSOURCE_PCLK_DIV32))

#define IS_LPUART_WORDLENGTH(__WORDLENGTH__)                (((__WORDLENGTH__) == LPUART_WORDLENGTH_7B)   ||\
                                                             ((__WORDLENGTH__) == LPUART_WORDLENGTH_8B))

#define IS_LPUART_STOPBITS(__STOPBITS__)                    (((__STOPBITS__) == LPUART_STOPBITS_1B)   ||\
                                                             ((__STOPBITS__) == LPUART_STOPBITS_2B))

#define IS_LPUART_PARITY(__PARITY__)                        (((__PARITY__) == LPUART_PARITY_NONE)   ||\
                                                             ((__PARITY__) == LPUART_PARITY_ODD)    ||\
                                                             ((__PARITY__) == LPUART_PARITY_EVEN)   ||\
                                                             ((__PARITY__) == LPUART_PARITY_0)      ||\
                                                             ((__PARITY__) == LPUART_PARITY_1))

#define IS_LPUART_MODE(__MODE__)                            (((__MODE__) == LPUART_MODE_TX)   ||\
                                                             ((__MODE__) == LPUART_MODE_RX)   ||\
                                                             ((__MODE__) == LPUART_MODE_TXRX))

#define IS_LPUART_WAKEUPMODE(__WAKEUPMODE__)                (((__WAKEUPMODE__) == LPUART_WAKEUPMODE_NONE)           ||\
                                                             ((__WAKEUPMODE__) == LPUART_WAKEUPMODE_STARTBIT)       ||\
                                                             ((__WAKEUPMODE__) == LPUART_WAKEUPMODE_ONEBYTENOCHECK) ||\
                                                             ((__WAKEUPMODE__) == LPUART_WAKEUPMODE_ONEBYTECHECK)   ||\
                                                             ((__WAKEUPMODE__) == LPUART_WAKEUPMODE_ADDRNOCHECK)    ||\
                                                             ((__WAKEUPMODE__) == LPUART_WAKEUPMODE_ADDRCHECK))

#define IS_LPUART_WAKEUPADDR(__WAKEUPADDR__)                ((__WAKEUPADDR__) <= 0xFFU)

#define IS_LPUART_ALL_IT(__INT__)                           ((((__INT__) & LPUART_IT_MASK) != 0U)   && \
                                                             (((__INT__) & ~LPUART_IT_MASK) == 0U))
                                                                                   
#define IS_LPUART_SINGLE_IT(__INT__)                        (((__INT__) == LPUART_IT_START)  ||\
                                                             ((__INT__) == LPUART_IT_MATCH)  ||\
                                                             ((__INT__) == LPUART_IT_RXOV)   ||\
                                                             ((__INT__) == LPUART_IT_FE)     ||\
                                                             ((__INT__) == LPUART_IT_PE)     ||\
                                                             ((__INT__) == LPUART_IT_TXE)    ||\
                                                             ((__INT__) == LPUART_IT_TC)     ||\
                                                             ((__INT__) == LPUART_IT_RX))
                                                         
#define IS_LPUART_ALL_FLAG(__FLAG__)                        ((((__FLAG__) & LPUART_FLAG_MASK) != 0U)   && \
                                                             (((__FLAG__) & ~LPUART_FLAG_MASK) == 0U))
                                                                     
#define IS_LPUART_SINGLE_FLAG(__FLAG__)                     (((__FLAG__) == LPUART_FLAG_STARTIF)    ||\
                                                             ((__FLAG__) == LPUART_FLAG_MATCHIF)    ||\
                                                             ((__FLAG__) == LPUART_FLAG_TXOVF)      ||\
                                                             ((__FLAG__) == LPUART_FLAG_RXF)        ||\
                                                             ((__FLAG__) == LPUART_FLAGR_RXOVIF)    ||\
                                                             ((__FLAG__) == LPUART_FLAG_FEIF)       ||\
                                                             ((__FLAG__) == LPUART_FLAG_PEIF)       ||\
                                                             ((__FLAG__) == LPUART_FLAG_TXEIF)      ||\
                                                             ((__FLAG__) == LPUART_FLAG_TCIF)       ||\
                                                             ((__FLAG__) == LPUART_FLAG_RXIF))

#define IS_LPUART_CLEAR_FLAG(__FLAG__)                      ((((__FLAG__) & LPUART_CLEAR_FLAG_MASK) != 0U)   && \
                                                             (((__FLAG__) & ~LPUART_CLEAR_FLAG_MASK) == 0U))
                                                              

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/

void LPUART_DeInit(LPUART_TypeDef* LPUARTx);
void LPUART_Init(LPUART_TypeDef* LPUARTx, LPUART_InitTypeDef *LPUART_InitStruct);
void LPUART_SetClockSource(LPUART_TypeDef* LPUARTx, uint32_t ClockSource);
void LPUART_SetBaudRate(LPUART_TypeDef* LPUARTx, uint32_t BaudRate, \
                        uint32_t WordLength, uint32_t Parity, uint32_t StopBit);
void LPUART_SetWordLength(LPUART_TypeDef* LPUARTx, uint32_t WordLength);
void LPUART_SetStopBit(LPUART_TypeDef* LPUARTx, uint32_t StopBit);
void LPUART_SetParity(LPUART_TypeDef* LPUARTx, uint32_t Parity);
void LPUART_SetWakeupMode(LPUART_TypeDef* LPUARTx, uint32_t WakeupMode);
void LPUART_SetWakeupAddr(LPUART_TypeDef* LPUARTx, uint32_t WakeupAddr);
void LPUART_SetReverseTxPolarity(LPUART_TypeDef* LPUARTx, FunctionalState NewState);
void LPUART_SetReverseRxPolarity(LPUART_TypeDef* LPUARTx, FunctionalState NewState);
void LPUART_TxCmd(LPUART_TypeDef* LPUARTx, FunctionalState NewState);
void LPUART_RxCmd(LPUART_TypeDef* LPUARTx, FunctionalState NewState);
void LPUART_DMACmd(LPUART_TypeDef* LPUARTx, FunctionalState NewState);
void LPUART_SendData(LPUART_TypeDef* LPUARTx, uint8_t Data);
uint8_t LPUART_ReceiveData(LPUART_TypeDef* LPUARTx);
void LPUART_ITConfig(LPUART_TypeDef* LPUARTx, uint32_t Int, FunctionalState NewState);
FlagStatus LPUART_GetFlagStatus(LPUART_TypeDef* LPUARTx, uint32_t Flag);
void LPUART_ClearFlag(LPUART_TypeDef* LPUARTx, uint32_t Flag);
FlagStatus LPUART_GetITStatus(LPUART_TypeDef* LPUARTx, uint32_t Int);


#endif
