/*
  ******************************************************************************
  * @file    HAL_RTC.c
  * @version V1.0.0
  * @date    2020
  * @brief   RTC HAL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the Real-Time Clock (RTC) peripheral:
  *           + Initialization functions
  *           + Time and Date configuration
  *           + Alarm configuration
  *           + WakeUp Timer configuration
  *           + TimeStamp configuration
  *           + Tampers configuration
  *           + Backup Data Registers configuration
  *           + RTC Tamper and TimeStamp Pins Selection
  *           + Interrupts and flags management
  ******************************************************************************
*/
#include "ACM32Fxx_HAL.h"

/*********************************************************************************
* Function    : HAL_RTC_Config
* Description : Initialize the RTC peripheral
* Input       : 
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
HAL_StatusTypeDef HAL_RTC_Config(RTC_ConfigTypeDef *hrtc)
{
#if (USE_FULL_ASSERT == 1)
    /* Check RTC Parameter */
    if (!IS_RTC_CLOCKSRC(hrtc->u32_ClockSource))         return HAL_ERROR;
    if (!IS_RTC_COMPENSATION(hrtc->u32_Compensation))    return HAL_ERROR;
#endif

    HAL_RTC_PCLK_ENABLE();
    
    /* RTC domain write enable */
    SCU->STOPCFG |= (1 << 0);

    PMU->CR1 |= RPMU_CR_RTCEN;

    switch (hrtc->u32_ClockSource)
    {
        case RTC_CLOCK_RC32K: 
        {
            PMU->ANACR |= RPMU_ANACR_RC32K_EN;
            while(!(PMU->ANACR & RPMU_ANACR_RC32K_RDY));

            PMU->CR1 &= ~RTC_CLOCK_XTL;
        }break;

        case RTC_CLOCK_XTL: 
        {
            PMU->ANACR = (PMU->ANACR & ~RPMU_ANACR_XTLDRV) | (RPMU_ANACR_XTLDRV_1 | RPMU_ANACR_XTLDRV_0);
            
            PMU->ANACR |= RPMU_ANACR_XTLEN;
            while(!(PMU->ANACR & RPMU_ANACR_XTLRDY));

            PMU->CR1 |= RTC_CLOCK_XTL;
        }break;

        default: break; 
    }

    if (hrtc->u32_CompensationValue) 
    {
        RTC->ADJUST = hrtc->u32_Compensation | hrtc->u32_CompensationValue;
    }

    HAL_RTC_PCLK_DISABLE();
    return HAL_OK;
}

/*********************************************************************************
* Function    : HAL_RTC_SetTime
* Description : Set RTC current time.
* Input       : fp_Time Pointer to Time structure.
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_SetTime(RTC_TimeTypeDef *fp_Time)
{
#if (USE_FULL_ASSERT == 1)
    /* Check RTC Parameter */
    if (!IS_RTC_HOUR(fp_Time->u8_Hours))     return;
    if (!IS_RTC_MIN(fp_Time->u8_Minutes))    return;
    if (!IS_RTC_SEC(fp_Time->u8_Seconds))    return;
#endif
    HAL_RTC_PCLK_ENABLE();
    /* Write-Protect Disable */
    RTC->WP = 0xCA53CA53;

    RTC->HOUR = fp_Time->u8_Hours;
    RTC->MIN  = fp_Time->u8_Minutes;
    RTC->SEC  = fp_Time->u8_Seconds;

    /* Write-Protect Enable */
    RTC->WP = 0;
    
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_GetTime
* Description : Get RTC current time.
* Input       : fp_Time Pointer to Time structure.
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_GetTime(RTC_TimeTypeDef *fp_Time)
{
    HAL_RTC_PCLK_ENABLE();
    fp_Time->u8_Hours   = RTC->HOUR;
    fp_Time->u8_Minutes = RTC->MIN;
    fp_Time->u8_Seconds = RTC->SEC;
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_SetDate
* Description : Set RTC current Date.
* Input       : fp_Date Pointer to Date structure.
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_SetDate(RTC_DateTypeDef *fp_Date)
{
#if (USE_FULL_ASSERT == 1)
    /* Check RTC Parameter */
    if (!IS_RTC_YEAR(fp_Date->u8_Year))          return;
    if (!IS_RTC_MONTH(fp_Date->u8_Month))        return;
    if (!IS_RTC_DAY(fp_Date->u8_Date))           return;
    if (!IS_RTC_WEEKDAY(fp_Date->u8_WeekDay))    return;
#endif
    HAL_RTC_PCLK_ENABLE();
    /* Write-Protect Disable */
    RTC->WP = 0xCA53CA53;

    RTC->YEAR  = fp_Date->u8_Year;
    RTC->MONTH = fp_Date->u8_Month;
    RTC->DATE  = fp_Date->u8_Date;
    RTC->WEEK  = fp_Date->u8_WeekDay;

    /* Write-Protect Enable */
    RTC->WP = 0;
    
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_GetDate
* Description : Get RTC current Date.
* Input       : fp_Date Pointer to Date structure.
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_GetDate(RTC_DateTypeDef *fp_Date)
{
    HAL_RTC_PCLK_ENABLE();
    fp_Date->u8_Year    = RTC->YEAR;
    fp_Date->u8_Month   = RTC->MONTH;
    fp_Date->u8_Date    = RTC->DATE;
    fp_Date->u8_WeekDay = RTC->WEEK;
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_AlarmConfig
* Description : Alarm Config
* Input       : fp_Alarm Pointer to ALarm structure.
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_AlarmConfig(RTC_AlarmTypeDef *fp_Alarm)
{
    uint32_t lu32_WeekDay;
    
#if (USE_FULL_ASSERT == 1)
    /* Check RTC Parameter */
    if (!IS_RTC_ALARM_MODE(fp_Alarm->u32_AlarmMode))        return;
    if (!IS_RTC_ALARM_INT(fp_Alarm->u32_AlarmInterrupt))    return;
    if (!IS_RTC_ALARM_DAY_MASK(fp_Alarm->u32_DayMask))      return;
    if (!IS_RTC_ALARM_HOUR_MASK(fp_Alarm->u32_HourMask))    return;
    if (!IS_RTC_ALARM_MIN_MASK(fp_Alarm->u32_MinMask))      return;
    
    if (fp_Alarm->u32_AlarmMode == RTC_ALARM_WEEK_MODE) 
    {
        if (!IS_RTC_ALARM_WEEKDAY(fp_Alarm->u32_AlarmWeek))    return;
    }
    else 
    {
        if (!IS_RTC_DAY(fp_Alarm->u32_AlarmDay))    return;
    }
    
    if (!IS_RTC_HOUR(fp_Alarm->u32_Hours))     return;
    if (!IS_RTC_MIN(fp_Alarm->u32_Minutes))    return;
    if (!IS_RTC_SEC(fp_Alarm->u32_Seconds))    return;
#endif
    HAL_RTC_PCLK_ENABLE();
    
    if (fp_Alarm->u32_AlarmMode == RTC_ALARM_WEEK_MODE) 
    {
        lu32_WeekDay = fp_Alarm->u32_AlarmWeek;
    }
    else 
    {
        lu32_WeekDay = fp_Alarm->u32_AlarmDay << RTC_ALARM_DAY_BIT_POSITION;
    }

    /* Coinfig Week/DayHourMinSec */
    RTC->ALM = fp_Alarm->u32_AlarmMode | lu32_WeekDay | fp_Alarm->u32_Hours << 16 | fp_Alarm->u32_Minutes << 8 | fp_Alarm->u32_Seconds;

    /* Interrupt Enable */
    if (RTC_ALARM_INT_ENABLE == fp_Alarm->u32_AlarmInterrupt) 
    {
        RTC->IE |= RTC_IE_ALM;
    }

    MODIFY_REG(RTC->CR, RTC_CR_ALM_MKSD, fp_Alarm->u32_DayMask); 
    
    MODIFY_REG(RTC->CR, RTC_CR_ALM_MSKH, fp_Alarm->u32_HourMask); 
    
    MODIFY_REG(RTC->CR, RTC_CR_ALM_MSKM, fp_Alarm->u32_MinMask); 
    
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_AlarmEnable
* Description : Alarm Enable
* Input       : 
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_AlarmEnable(void)
{
    HAL_RTC_PCLK_ENABLE();
    RTC->CR |= RTC_CR_ALM_EN;
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_AlarmDisable
* Description : Alarm Disable
* Input       : 
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_AlarmDisable(void)
{
    HAL_RTC_PCLK_ENABLE();
    RTC->CR &= ~RTC_CR_ALM_EN;
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_Tamper
* Description : Temper1 use PC13Temper2 use PA0
* Input       : 
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_Tamper(enum_Temper_t fe_Temper, RTC_TemperTypeDef *fp_Temper)
{
#if (USE_FULL_ASSERT == 1)
    /* Check RTC Parameter */
    if (!IS_RTC_TEMP_EDGE(fp_Temper->u32_TemperEdge))             return;
    if (!IS_RTC_TEMP_INT(fp_Temper->u32_InterruptEN))             return;
    if (!IS_RTC_TEMP_CLEAR_BACKUP(fp_Temper->u32_ClearBackup))    return;
    if (!IS_RTC_TEMP_FILTER(fp_Temper->u32_Filter))               return;
#endif
    HAL_RTC_PCLK_ENABLE();
    switch (fe_Temper)
    {
        case RTC_TEMPER_1: 
        {
            PMU->IOCR  &= ~(1<<6);  // Configure PC13 as digital IO   
			if(fp_Temper->u32_TemperEdge ==RTC_TEMP_EDGE_FALLING) 
			{
				// Configure PC13 pull up
				PMU->IOCR |= 1<<0;   
				PMU->IOCR &=~(1<<1);
			}
			else
			{
				// Configure PC13 pull down
				PMU->IOCR &= ~(1<<0);
				PMU->IOCR |=(1<<1);	
			}
            PMU->IOSEL |=  0x02;  // Configure PC13 as tamper function    
            
            /* Clear Config */
            RTC->CR &= ~(RTC_CR_TAMP1RCLR | RTC_CR_TAMP1FCLR | RTC_CR_TAMP1FLTEN | RTC_CR_TAMP1FLT | RTC_CR_TS1EDGE | RTC_CR_TAMPFLTCLK);
            /* Edge select */
            RTC->CR |= fp_Temper->u32_TemperEdge ? RTC_CR_TS1EDGE : 0x00;
            /* Auto clear backup register */
            if (fp_Temper->u32_ClearBackup) 
            {
                RTC->CR |= fp_Temper->u32_TemperEdge ? RTC_CR_TAMP1FCLR : RTC_CR_TAMP1RCLR;
            }
            /* Temper filter */
            if (fp_Temper->u32_Filter) 
            {
                if (fp_Temper->u32_Filter == RTC_TEMP_FILTER_512_RTCCLK) 
                {
                    RTC->CR |= RTC_CR_TAMPFLTCLK;
                }
                else 
                {
                    RTC->CR |= (fp_Temper->u32_Filter - 2) << 13;
                }
            }  
            
            RTC->CR |= RTC_CR_TAMP1EN;  
            System_Delay(2000);       
            RTC->SR |= (RTC_SR_STP1FIE|RTC_SR_STP1RIE);   
            RTC->IE &= (~(RTC_IE_STP1FIE|RTC_IE_STP1RIE));      
            
            /* Put Temper Interrupt enable here !!!*/   
            if (fp_Temper->u32_InterruptEN) 
            {
                RTC->IE |= fp_Temper->u32_TemperEdge ? RTC_IE_STP1FIE : RTC_IE_STP1RIE;
            }               
 
        }break;
        
        case RTC_TEMPER_2:
        {
			SCU->PABADS &= ~(1<<0);
			if(fp_Temper->u32_TemperEdge ==RTC_TEMP_EDGE_FALLING) 
			{
				// Configure PA0 pull up
				SCU->PABPUR |=  1<<0;   
				SCU->PABPDR &=~(1<<0);
			}
			else
			{
				// Configure PA0 pull down
				SCU->PABPUR &= ~(1<<0);   
				SCU->PABPDR |= 1<<0;
			}
            /* Clear Config */
            RTC->CR &= ~(RTC_CR_TAMP2RCLR | RTC_CR_TAMP2FCLR | RTC_CR_TAMP2FLTEN | RTC_CR_TAMP2FLT | RTC_CR_TS2EDGE | RTC_CR_TAMPFLTCLK);
            /* Edge select */
            RTC->CR |= fp_Temper->u32_TemperEdge ? RTC_CR_TS2EDGE : 0x00;
            /* Auto clear backup register */
            if (fp_Temper->u32_ClearBackup) 
            {
                RTC->CR |= fp_Temper->u32_TemperEdge ? RTC_CR_TAMP2FCLR : RTC_CR_TAMP2RCLR;
            }
            /* Temper filter */
            if (fp_Temper->u32_Filter) 
            {
                if (fp_Temper->u32_Filter == RTC_TEMP_FILTER_512_RTCCLK) 
                {
                    RTC->CR |= RTC_CR_TAMPFLTCLK;
                }
                else 
                {
                    RTC->CR |= (fp_Temper->u32_Filter - 2) << 19;
                }
            }
            
            RTC->CR |= RTC_CR_TAMP2EN;  
            System_Delay(2000);     
            RTC->SR |= (RTC_SR_STP2FIE|RTC_SR_STP2RIE);   
            RTC->IE &= (~(RTC_IE_STP2FIE|RTC_IE_STP2RIE));  
            
            /* Temper Interrupt */
            if (fp_Temper->u32_InterruptEN) 
            {
                RTC->IE |= fp_Temper->u32_TemperEdge ? RTC_IE_STP2FIE : RTC_IE_STP2RIE;
            }

        }break;

        default: break; 
    }
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_TamperEnable
* Description : 
* Input       : 
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_TamperEnable(enum_Temper_t fe_Temper)
{
    HAL_RTC_PCLK_ENABLE();
    if (fe_Temper == RTC_TEMPER_1) 
    {
        RTC->CR |= RTC_CR_TAMP1EN;
    }
    else 
    {
        RTC->CR |= RTC_CR_TAMP2EN;
    }
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_TamperDisable
* Description : 
* Input       : 
* Outpu       : 
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
void HAL_RTC_TamperDisable(enum_Temper_t fe_Temper)
{
    HAL_RTC_PCLK_ENABLE();
    if (fe_Temper == RTC_TEMPER_1) 
    {
        RTC->CR &= ~RTC_CR_TAMP1EN;
    }
    else 
    {
        RTC->CR &= ~RTC_CR_TAMP2EN;
    }
    HAL_RTC_PCLK_DISABLE();
}

/***************************************************************************************************
* Function    : HAL_RTC_Standby_Wakeup
* Description : wakeup source select
* Input       : fu32_Edge 0: Rising edge
*                         1: Falling edge
*               fe_Wakeup  : wakeup source select, STANDBY_WAKEUP_RISING, STANDBY_WAKEUP_FALLING  
* Outpu       : 
* Author      : Chris_Kyle                         Date : 2021   
*******************************************************************************************************/
void HAL_RTC_Standby_Wakeup(enum_WKUP_t fe_Wakeup, uint32_t fu32_Edge)
{
    HAL_RTC_PCLK_ENABLE();
    switch (fe_Wakeup)
    {
        case RTC_WAKEUP_WKUP1: 
        case RTC_WAKEUP_WKUP2: 
        case RTC_WAKEUP_WKUP3: 
        case RTC_WAKEUP_WKUP4: 
        case RTC_WAKEUP_WKUP5: 
        case RTC_WAKEUP_WKUP6:   
        {
            /* Clear flagsStandby Enable */
            PMU->CR1 |= RPMU_CR_STB_EN | RPMU_CR_CWUF | RPMU_CR_CSBF;

            /* Wakeup IO Filter Enable */
            PMU->CR1 |= fe_Wakeup << 8;
            /* Wakeup IO Enable */
            PMU->CR1 |= fe_Wakeup;

            if (fe_Wakeup == RTC_WAKEUP_WKUP2) 
            {
                /* PC13 */
                PMU->IOCR &= ~0x40;  // must configure PC13 as digital function     
            }
            
            if (fu32_Edge) 
            {
                PMU->CR2 |= fe_Wakeup >> 16;
            }
            else 
            {
                PMU->CR2 &= ~(fe_Wakeup >> 16);   
            }

            PMU->CR1 |= RPMU_CR_CWUF; // clear wakeup flag     
            System_Enter_Standby_Mode();              
        }break;

        case RTC_WAKEUP_STAMP2:
        case RTC_WAKEUP_STAMP1:
        case RTC_WAKEUP_32S:
        case RTC_WAKEUP_SEC:
        case RTC_WAKEUP_MIN:
        case RTC_WAKEUP_HOUR:
        case RTC_WAKEUP_DATE:     
        {
            /* Clear flagsStandby Enable */
            PMU->CR1 |= RPMU_CR_STB_EN | RPMU_CR_CWUF | RPMU_CR_CSBF;

            RTC->SR |= fe_Wakeup;
            RTC->IE |= fe_Wakeup;

            System_Enter_Standby_Mode();  
        }break;

        default: break; 
    }
    HAL_RTC_PCLK_DISABLE();
}

/*********************************************************************************
* Function    : HAL_RTC_GetStandbyStatus
* Description : Check MCU have entered standby mode
* Input       : 
* Outpu       : 0: Not Enter Standby Mode
                1: Entered Standby Mode
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
bool HAL_RTC_Get_StandbyStatus(void)
{
    bool ret;
    HAL_RTC_PCLK_ENABLE();
    if (PMU->SR & RPMU_SR_SBF) 
    {
        ret = true;
    }
    else 
    {
        ret = false;
    }
    HAL_RTC_PCLK_DISABLE();
    return ret;
}

/*********************************************************************************
* Function    : HAL_RTC_Get_StandbyWakeupSource
* Description : Get MCU Standby Wakeup Source
* Input       : 
* Outpu       : RTC_WAKEUP_SOURCE_BORWUF
                RTC_WAKEUP_SOURCE_IWDTWUF
                RTC_WAKEUP_SOURCE_RSTWUF
                RTC_WAKEUP_SOURCE_RTCWUF
                RTC_WAKEUP_SOURCE_WKUP6
                RTC_WAKEUP_SOURCE_WKUP5
                RTC_WAKEUP_SOURCE_WKUP4
                RTC_WAKEUP_SOURCE_WKUP3
                RTC_WAKEUP_SOURCE_WKUP2
                RTC_WAKEUP_SOURCE_WKUP1
* Author      : Chris_Kyle                         Data : 2020
**********************************************************************************/
uint32_t HAL_RTC_Get_StandbyWakeupSource(void)
{
    uint32_t temp;
    HAL_RTC_PCLK_ENABLE();
    temp = PMU->SR;
    HAL_RTC_PCLK_DISABLE();
    return temp;
}

/*********************************************************************************
* Function    : HAL_RTC_BORResetConfig
* Description : BOR Reset config
* Input       : Voltage: BOR Voltage
* Output      : None
* Author      : Sam                         Data : 2023
**********************************************************************************/
void HAL_RTC_BORResetConfig(uint32_t Voltage)
{
#if (USE_FULL_ASSERT == 1)
    /* Check Parameter */
    if (!IS_RPMU_BOR_VOLTAGE(Voltage))  return;
#endif

    HAL_RTC_PCLK_ENABLE();
    
    /* RTC domain write enable */
    SCU->STOPCFG |= (1 << 0);
    
    /* BOR ѹѡ */
    PMU->ANACR |= (Voltage << 24) | (1 << 23);
    
    System_Delay(1000); //delay 1us
    
    /* BOR λʹ */
    PMU->CR1 |= (1 << 12);
    
    
}
